#!/usr/bin/bash

set -euo pipefail

BSS_JSON=$(cray bss bootparameters list --format json)
BSS_CUSTOM_JSON=$(jq '
  [
    .[]
    | select(.["cloud-init"]["meta-data"]["shasta-role"])
    | {
        id: (.params | capture("metal\\.server=[^ ]*/boot-images/(?<id>[^/]+)/rootfs").id? // null),
        hn: (.params | capture("hostname=(?<hostname>[^ ]+)").hostname? // null),
        role: .["cloud-init"]["meta-data"]["shasta-role"],
        xname: .["cloud-init"]["meta-data"]["xname"]
      }
  ]
  | group_by(.role)
  | map({
      "shasta_role": .[0].role,
      nodes: map({ "ims_image_id": .id, hostname: .hn, "xname": .xname })
    })
' <<< "$BSS_JSON")

CFS_JSON=$(cray cfs components list --format json)
CFS_CUSTOM_JSON=$(jq '[ .[] | {xname: .id, current_cfs_config: .desiredConfig}]' <<< "$CFS_JSON")

CRAY_CATALOG_CUSTOM_JSON="$(
  kubectl -n services get cm cray-product-catalog -o jsonpath='{.data.csm}' \
    | yq4 -o=json \
    | jq '
        to_entries 
        | map(
            if .value.images 
            then
              { 
                key: .key,
                value: {
                  images: (
                    .value.images
                    | to_entries
                    | map(select(.key | test("^(secure-kubernetes|secure-storage-ceph)")))
                    | from_entries
                  )
                }
              }
            else empty end
          )
        | map(select(.value.images | length > 0))
        | from_entries
      '
)"

# Merge BSS, CFS, cray catalog data
MERGED_JSON=$(
  jq --argjson cfs "$CFS_CUSTOM_JSON" '
    map(
      .nodes |= map(
        . as $s | . + ( ($cfs[] | select(.xname == $s.xname) | {current_cfs_config}) // {} )
      )
    )
  ' <<< "$BSS_CUSTOM_JSON"
)

# Compose final doc
FINAL_JSON=$(jq -n \
  --argjson management_nodes "$MERGED_JSON" \
  --argjson csm_versions "$CRAY_CATALOG_CUSTOM_JSON" \
  '{management_nodes: $management_nodes, csm_versions: $csm_versions}'
)

echo "${FINAL_JSON}" | tee system-info.json